//Event class header file
#pragma once


//******************************************************************************************************
//**** PROJECT HEADER FILES AND CLASS NAME DECLARATIONS
//******************************************************************************************************
#include "SchedulerBuild.h"
#include "AssertableRWSemaphore.h"
#include "BufferedFile.h"
#include "CLVListItem.h"
class EventAddOnData;
class EventWindow;
class EventSet;


//******************************************************************************************************
//**** TYPE DEFINITIONS AND ASSOCIATED CONSTANTS
//******************************************************************************************************
enum ConstructorResult {NoError, FileOpenError, FileError, BugError};
enum StatusType {WaitingForPrewarn, WaitingForEvent, WaitingForNag, DoneWithEvent};
enum PeriodicityType {Once, Annually, Monthly, Weekly, Daily, Weekdays, Weekends, MultiAnnually,
	MultiMonthly,MultiWeekly, MultiDaily, SubDaily, NumberOfPeriodicityTypes};
enum TimingType {ByDate, ByDayOfMonth, ByDayOfWeek, NumberOfTimingTypes};
enum TimeIntervalType {Months, Weeks, Days, Hours, Minutes};


//******************************************************************************************************
//**** GLOBAL VARIABLES AND ASSOCIATED CONSTANTS
//******************************************************************************************************
const time_t WeekLength = 604800;
const time_t DayLength = 86400;
const time_t HourLength = 3600;
const time_t MinLength = 60;
extern bool AllowableTypeCombination[NumberOfPeriodicityTypes][NumberOfTimingTypes];
extern const time_t IntervalLength[5];		//Lengths of the intervals by TimeIntervalType.  Months not
											//allowed.
extern const char* DayNames[7];
extern const char* MonthNames[12];


//******************************************************************************************************
//**** FUNCTION DECLARATIONS
//******************************************************************************************************
int8 DetermineWeekOfMonth(int8 DayOfMonth, int8 DayOfWeek);
int8 DaysInMonth(int32 Month, int32 Year);
char* GetNagOrPrewarnMessage(char PrewarnOrNag, char* Name, time_t ActualEventTime, bool ApplyTime,
	char* Message);
time_t GetLast12AM(time_t When);
int32 GetSecondsIntoDay(time_t When);
char tolower(char x);


//******************************************************************************************************
//**** CLASS AND STRUCTURE DECLARATIONS AND ASSOCIATED CONSTANTS
//******************************************************************************************************
class _SCHIMPEXP Event : public CLVListItem
{
	public:
		//Constructor and destructor
		Event(EventSet* OwnerSet);
		Event(EventSet* OwnerSet, BufferedFile* File, uint8 FileVersionIndex, ConstructorResult* Result,
			bool* VersionKnown);
		Event(EventSet* OwnerSet, BMessage* PasteMessage, bool* Success);
		~Event();

		//Get values functions
		bool GenerateMessageForEvent(BMessage* EventMessage);
		const char* GetName();
		time_t GetNextEventTime();			//Gets the time of the next event, including prewarn and
													//nag events
		time_t GetActualEventTime();		//Gets the time of the actual event, not considering
													//prewarn or nag events		
		bool GetActive();					//Gets the active/inactive state for this event
		StatusType GetStatus();			//Gets the status of the event
		PeriodicityType GetPeriodType();	//Gets the #1 selector (period type)
		uint32 GetMultiple();				//Gets the number of time increments between multimonthly,
													//etc. events
		TimingType GetTimingType();		//Gets the #2 selector (exact timing type)
		time_t GetEventTime();				//Gets the #3 selector (ByDate)
		int8 GetDayOfMonth();				//Gets the #3 selector (ByDayOfMonth)
		int8 GetWeekOfMonth();				//Gets the #3 selector (ByDayOfWeek: Annually, Monthly,
													//Multiannually, Multimonthly)
		int8 GetDayOfWeek()	;			//Gets the #3 selector (ByDayOfWeek: Always)
		int8 GetMonth();					//Gets the #3 selector (ByDayOfWeek: Anually,
													//Multiannually)
		time_t GetMultiStartTime();		//Gets the #4 selector (Starting time/phase for
													//Multi-anything events)
		bool GetApplyTime();				//Gets the #5 selector (Apply time to the day?)
		int32 GetTimeToApply();			//Gets the #5 selector (Time to apply in the day)
		TimeIntervalType GetSubIntType();	//Gets the #6 selector (subinterval type).  Only hours or
													//minutes are allowed
		bool GetApplyStop();				//Should the events stop (for the night, etc.)?
		int32 GetStopTime();				//Gets the #6 selector (time the events should stop, in 
													//seconds since midnight)
		uint32 GetSelectedHandler();				//Gets the handler selected to deal with this event
		EventAddOnData* GetAddOnData();				//Gets the add-on data for the currently selected add-on
		EventAddOnData* GetAddOnData(uint32 AO);	//Gets the add-on data for a specific add-on (0-N)
		bool GetApplyPrewarn();					//Gets ApplyPrewarn flag
		uint32 GetPrewarnTime();					//Gets how far in advance prewarning starts
		TimeIntervalType GetPrewarnType();			//Gets the time interval type for prewarning start time
		bool GetApplyPrewarnRepeat();				//Gets flag for whether prewarning is repeated
		uint32 GetPrewarnRepeatTime();				//Gets how far apart the prewarn warnings are
		TimeIntervalType GetPrewarnRepeatType();	//Gets the time interval type for prewarning repeats
		const char* GetPrewarnMessage();			//Gets the prewarning message
		bool GetApplyNag();					//Gets ApplyNag flag
		uint32 GetNagRepeatTime();				//Gets how far apart the nag messages are
		TimeIntervalType GetNagRepeatType();	//Gets the time interval type for nag messages
		const char* GetNagMessage();			//Gets the nag message
		bool GetDeliverIfMissed();				//Gets the DeliverIfMissed flag
		bool GetDeleteOnceDelivered();			//Gets the DeleteOnceDelivered flag
		const char* GetComment();				//Gets the comment
		EventWindow* GetWindow();				//Gets the window for this event
		EventSet* GetOwner();					//Gets the set that owns this event
		BRect GetWindowFrame();							//Gets the window frame used by this event's window
		int32 GetIcon(BBitmap** LargeIcon, BBitmap** MiniIcon);		//Gets pointers to the icons
		int32 GetTabShown();
		bool GetShouldWindowOpen();
		bool GetCanBeDeleted();

		//Set values functions
		void SetName(const char* NewName);		//Sets the name of the event
		void SetActive(bool State);				//Sets the active/inactive state for this event
		void SetPeriodType(PeriodicityType);	//Sets the #1 selector (period type); doesn't affect window
		void SetMultiple(uint32 M);				//Sets the number of time increments between multimonthly,
												//etc. events
		void SetTimingType(TimingType TT);				//Sets the #2 selector (exact timing type)
		void SetEventTime(time_t ET, bool UseHMS);		//Sets the #3 selector (ByDate).  UseHMS indicates
														//whether or not to use the hours, minutes, seconds
														//information
		void SetDayOfMonth(int8 DOM);					//Sets the #3 selector (ByDayOfMonth)
		void SetWeekOfMonth(int8 WOM);					//Sets the #3 selector (ByDayOfWeek: Annually,
														//Monthly, Multiannually, Multimonthly)
		void SetDayOfWeek(int8 DOW);					//Sets the #3 selector (ByDayOfWeek: Always)
		void SetMonth(int8 M);							//Sets the #3 selector (ByDayOfWeek: Anually,
														//Multiannually)
		void SetMultiStartTime(time_t MST);				//Sets the #4 selector (Starting time/phase for
														//Multi-anything events)
		void SetApplyTime(bool AT);						//Sets the #5 selector (Apply time to the day?)
		void SetTimeToApply(int32 SID);					//Sets the #5 selector (Time to apply in the day)
		void SetSubIntType(TimeIntervalType TT);		//Sets the #6 selector (subinterval type).  Only
														//hours or minutes are allowed
		void SetApplyStop(bool AS);						//Sets the #6 selector (Should the events stop?)
		void SetStopTime(int32 SID);					//Sets the #6 selector (Time to stop events)
		void SetSelectedHandler(uint32 AddOnIndex);		//Sets the handler selected to deal with this
														//event, calls that handler to initialize its data
														//if necessary, then draw its view
		void SetApplyPrewarn(bool AP);					//Sets ApplyPrewarn flag
		void SetPrewarnTime(uint32 HowFar);				//Sets how far in advance prewarning starts
		void SetPrewarnType(TimeIntervalType T);		//Sets the time interval type for prewarning start
														//time
		void SetApplyPrewarnRepeat(bool R);				//Sets flag for whether prewarning is repeated
		void SetPrewarnRepeatTime(uint32 T);			//Sets how far apart the prewarn warnings are
		void SetPrewarnRepeatType(TimeIntervalType T);	//Sets the time interval type for prewarning
														//repeats
		void SetPrewarnMessage(const char* Msg);		//Sets the prewarning message
		void SetApplyNag(bool AN);						//Sets ApplyNag flag
		void SetNagRepeatTime(uint32 T);				//Sets how far apart the nag messages are
		void SetNagRepeatType(TimeIntervalType T);		//Sets the time interval type for nag messages
		void SetNagMessage(const char* Msg);			//Sets the nag message
		void SetDeliverIfMissed(bool DIM);				//Sets the DeliverIfMissed flag
		void SetDeleteOnceDelivered(bool DOD);			//Sets the DeleteOnceDelivered flag
		void SetComment(const char* Com);				//Sets the comment
		void SetWindowFrame(BRect Frame);				//Sets the window frame used by the event window
		void SetIcon(int32 IconNum, BBitmap* LargeIcon, BBitmap* MiniIcon);
														//Sets the icons; Event now owns them
		void SetTabShown(int32 index);

		//Action functions
		void OpenWindow();						//Open a window for this event
		bool SaveData(BufferedFile* File);				//Save this event into the data file
		void DealWithEvent(bool ConsiderEventsMissed);
		void SetSkipNag();
		void EventDelivered();							//Event has been delivered; increment it
		void FixRegistrationReminderContent();
		void SetModified();


		//CLVListItem stuff
		static int Compare(const CLVListItem* a_item1, const CLVListItem* a_item2, int32 key_column);
		virtual void DrawItemColumn(BView* owner, BRect item_column_rect, int32 column_index, bool complete);
		virtual void DrawItem(BView* owner, BRect itemRect, bool complete);
		virtual void Update(BView *owner, const BFont *font);

		//Used to protect the event and associated set from changes
		AssertableRWSemaphore* GetProtectionSemaphore();

		//Grant friend access
		friend class EventWindow;
		friend class EventView;
		friend class IconView;

	private:
		//FBC protection
		virtual void Reserved1();
		virtual void Reserved2();
		virtual void Reserved3();
		virtual void Reserved4();
		virtual void Reserved5();
		virtual void Reserved6();
		int32 Reserved7;
		int32 Reserved8;
		int32 Reserved9;
		int32 Reserved10;
		int32 Reserved11;
		int32 Reserved12;
		
		/*Selector interaction:
			For ByDate, user only sees SelectEventTime in when field, but SelectEventTime, while
				retaining hours, minutes, and seconds information, also sets SecondsIntoDay.  ApplyTime
				checkbox is used.
			For ByDayOfMonth and ByDayOfWeek, SelectEventTime has no effect.  ApplyTime and SecondsIntoDay
				in time field are what user sees.
			For ByTimeOfDay, ApplyTime and SecondsIntoDay are used, SelectEventTime is ignored.

			PeriodType -> Available set of TimingTypes

	  		ByDate
				SelectEventTime --------------------\___
				ApplyTime/SecondsIntoDay -----------/	 \
														 |
			ByDayOfMonth								 |
				DayOfMonth, Month ------------------\____|--> Date and time
				ApplyTime/SecondsIntoDay -----------/	 |			\
														 |			 \--> ActualEventTime
			ByDayOfWeek									 |						\
				DayOfWeek, WeekInMonth, Month ------\____|						 \--->NextEventTime
				ApplyTime/SecondsIntoDay -----------/	 |
			ByTimeOfDay									 |
				ApplyTime/SecondsIntoDay ----------------/
		*/

		char* Name;
		int32 IconNum;						//Standard icons, indexed.  -1 = none, -2 = custom
		BBitmap* CustomLargeIcon;
		BBitmap* CustomMiniIcon;
		time_t NextEventTime;				//Time of the next event, including prewarn events, nag events,
											//etc.  Includes info from SecondsIntoDay if ApplyTime is true
		time_t ActualEventTime;				//Time of the actual event, derived from selectors.  Includes
											//info from SecondsIntoDay if ApplyTime is true
		bool Active;						//Should reminders be delivered for this event?
		StatusType Status;					//Status of the event
		//#1 selector
			PeriodicityType PeriodType;		//The type of periodicity mechanism used
			int32 Multiple;					//Number of months between MultiMonthly events, etc.
		//#2 selector
			TimingType TimeType;			//The type of exact timing mechanism used
			TimingType RememberType;		//If PeriodType changes and invalidates TimingType, remember
											//TimingType so if PeriodType changes again and invalidates
											//TimingType, I can switch back to this one
		//#3 selector						//The unused fields are preserved for if the user switches
											//back
			//Used by ByDate
				time_t SelectEventTime;		//Time of the actual event.  Includes info from SecondsIntoDay
											//if ApplyTime is true
			//Used by ByDayOfMonth
				int8 DayOfMonth;			//1-31; 32=Last, 33=2nd to last, ... 38=7th to last
			//Used by ByDayOfWeek
				int8 WeekOfMonth;			//0-3; 4=Last, 5=2nd to last	Used by Annually, Monthly,
											//								Multiannually, Multimonthly
				int8 DayOfWeek;				//0-6 = Sunday-Saturday			Always used with ByDayOfWeek
			//Used by ByDayOfMonth or ByDayOfWeek
				int8 Month;					//0-11 = January-December		Used by Annually and
											//								Multiannually
		//#4 selector: time_t for starting event, used to establish phase for multitimeinterval events.
		//This is not guaranteed to synch with the other selectors.  If it doesn't synch, start with the
		//first valid time after MultiStartTime.
			time_t MultiStartTime;
		//#5 selector: Time may be applied to ByDate, ByDayOfMonth, or ByDayOfWeek; required for
		//ByTimeOfDay
			bool ApplyTime;					//Should time count or just make it midnight? (true/false)
			int32 SecondsIntoDay;			//0-86399
		//#6 selector: Used by Sub-daily periodicity type
			//Use MultiStartTime for starting time - can be a plain time or on a specific date
			//Use Multiple for number of time increments for delay
			TimeIntervalType SubIntType;	//Sub-daily interval type (only use hours or minutes)
			bool ApplyStop;					//Should the events stop (for the night, etc.)
			int32 StopTime;					//0-86399 seconds into day
		//Events will be handled by add-ons
			EventAddOnData** AddOnData;		//The data for the add-on.  This points to an array of pointers
											//to add-on data, one for each add-on so as the user switches
											//add-ons to handle the event, how the event is handled by that
											//add-on will be remembered.
			uint32 SelectedHandler;			//The handler number to be used (0-N).  NOTE: when writing the
											//data file, use the add-on's name, NOT the number, as it
											//could change.
		//Prewarning before the event
			bool ApplyPrewarn;					//Should the user receive a prewarning?
			int32 PrewarnTime;					//How far in advance (number of months, weeks, days, etc.)
			TimeIntervalType PrewarnType;		//Is it months, weeks, days, hours, or minutes?
			bool ApplyPrewarnRepeat;			//Should the prewarning be repeated?
			uint32 PrewarnRepeatTime;			//How many months, weeks, days, hours, or minutes between
												//prewarning
			TimeIntervalType PrewarnRepeatType;	//Is it months, weeks, days, hours, or minutes between
												//repeats?
			char* PrewarnMessage;
		//Nagging after the event
			bool ApplyNag;					//Should the user be nagged until the event is dismissed?
											//Mutually exclusive with DeliverIfMissed
			uint32 NagRepeatTime;			//How many months, weeks, days, hours, or minutes between nags
			TimeIntervalType NagRepeatType;	//Is it months, weeks, days, hours, or minutes between
											//repeats?
			char* NagMessage;
		bool DeliverIfMissed;				//If the computer is off when this event occurs, should I
											//deliver it late?
		bool DeleteOnceDelivered;			//Should the event be deleted after delivery?  Mutually
											//exclusive with ApplyNag (true/false)
		char* Comment;

		BRect WindowFrame;
		bool ShouldWindowOpen;				//Only used after loading from a file to determine whether
											//when the set was saved this window was open and thus, should
											//it be opened now?
		int32 ShownTab;
		EventWindow* Window;				//If a window is open to edit this event, this points to it,
											//or NULL.
		EventSet* Owner;					//Set that owns this event
		float TextOffset;					//Offset to the text within list items
		bool SkipNag;						//User clicked Skip nag after event was delivered
		bool SkipPrewarn;					//User clicked Skip prewarn after event was delivered
		bool CanBeDeleted;

		//Utility functions:
		void UpdateEventTime();
			void UET_EvalDOM(tm &EventBDT);
			void UET_EvalDOW_MonthFixed(tm &EventBDT);
			void UET_EvalTodayOrTomorrow(time_t Now);
			void UET_EvalNextSubDaily(time_t Now);
		void IncrementPrewarn();						//Increment the prewarning
		void IncrementNag();							//Increment the nag
		void DeliverPrewarnMessage();
		void DeliverNagMessage();
		void UpdateViews();
};
